/*
 * @(#)AMApplicationModel.java  1.0  2006-12-05
 *
 * Copyright (c) 2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */
package ch.hslu.cm.am;

import ch.hslu.cm.am.diagram.*;
import ch.hslu.cm.app.action.Arrangeable;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import org.jhotdraw.app.*;
import org.jhotdraw.draw.*;
import org.jhotdraw.draw.action.*;
import org.jhotdraw.util.*;
import ch.randelshofer.html.*;
import org.jhotdraw.app.action.ActionUtil;
import org.jhotdraw.app.action.file.ExportFileAction;
import org.jhotdraw.app.action.view.ToggleViewPropertyAction;
import org.jhotdraw.app.action.view.ViewPropertyAction;
import org.jhotdraw.draw.tool.ConnectionTool;
import org.jhotdraw.draw.tool.CreationTool;
import org.jhotdraw.draw.tool.TextAreaCreationTool;
import org.jhotdraw.draw.tool.Tool;
import org.jhotdraw.gui.JFileURIChooser;
import org.jhotdraw.gui.URIChooser;
import org.jhotdraw.gui.filechooser.ExtensionFileFilter;

/**
 * The application model is used instantiate an activity modeling project
 * (AMView) within an Application, and to configure the menu bars and
 * toolbars.
 *
 *
 * @author Werner Randelshofer, Florian Padrun
 * @version 1.0 2006-06-18 Created.
 */
public class AMApplicationModel extends DefaultApplicationModel {

    /**
     * The scale factors are used for zooming a DrawingView.
     */
    /*private*/ final static double[] scaleFactors = {5, 4, 3, 2, 1.5, 1.25, 1, 0.75, 0.5, 0.25, 0.10};

    /**
     * The ToolButtonListener is used to activate a drawing Tool, when a button
     * in the toolbar is selected.
     */
    private static class ToolButtonListener implements ItemListener {

        private Tool tool;
        private DrawingEditor editor;

        public ToolButtonListener(Tool t, DrawingEditor editor) {
            this.tool = t;
            this.editor = editor;
        }

        @Override
        public void itemStateChanged(ItemEvent evt) {
            if (evt.getStateChange() == ItemEvent.SELECTED) {
                editor.setTool(tool);
            }
        }
    }
    /**
     * This editor is shared by all projects. This allows to have a single
     * set of toolbars for all open DrawingViews.
     * Note: Some Application's, like the SDIApplication enforce non-sharing of
     * DrawingEditors. That is, each View has its own DrawingEditor.
     * That's why the shared editor is lazily created the first time it is needed.
     */
    private DefaultDrawingEditor sharedEditor;
    /**
     * This HashMap holds actions shared by all projects.
     */
    private HashMap<String, Action> actions;

    /** Creates a new instance. */
    public AMApplicationModel() {
    }

    /**
     * Initializes the application.
     * We create here actions that are shared by all projects.
     *
     * @param a The Application that needs to be initalized.
     */
    @Override
    public ActionMap createActionMap(Application a, View v) {
        ResourceBundleUtil amLabels = ResourceBundleUtil.getBundle("ch.hslu.cm.am.Labels");
        ResourceBundleUtil cmLabels = ResourceBundleUtil.getBundle("ch.hslu.cm.Labels");
        AbstractAction aa;

        ActionMap am = super.createActionMap(a, v);

        // XXX - We don't have an writeHTML function because there is no AMApplet.
        // am.put(ExportFileAction.ID,aa=new ExportFileAction(a, v));

        am.put("toggleOptionsPanel", aa = new ToggleViewPropertyAction(
                a, null, "optionsVisible"));
        cmLabels.configureAction(aa, "options");
        am.put("toggleGrid", aa = new ToggleViewPropertyAction(
                a, null, "gridVisible"));
        cmLabels.configureAction(aa, "view.grid");
        am.put("pArrangeHorizontal", aa = new ViewPropertyAction(
                a, null, "arrangement", Arrangeable.Arrangement.HORIZONTAL));
        cmLabels.configureAction(aa, "view.arrangeHorizontal");
        am.put("pArrangeVertical", aa = new ViewPropertyAction(
                a, null, "arrangement", Arrangeable.Arrangement.VERTICAL));
        cmLabels.configureAction(aa, "view.arrangeVertical");
        am.put("pArrangeCascade", aa = new ViewPropertyAction(
                a, null, "arrangement", Arrangeable.Arrangement.CASCADE));
        cmLabels.configureAction(aa, "view.arrangeOverlap");
        for (double sf : scaleFactors) {
            am.put((int) (sf * 100) + "%",
                    aa = new ViewPropertyAction(a, null, "scaleFactor",
                    Double.TYPE, new Double(sf)));
            aa.putValue(Action.NAME, (int) (sf * 100) + " %");

        }
        return am;
    }

    /**
     * Returns the shared DrawingEditor. If the shared DrawingEditor does
     * not exist yet, it is created by this method.
     */
    public DefaultDrawingEditor getSharedEditor() {
        if (sharedEditor == null) {
            sharedEditor = new DefaultDrawingEditor();
        }
        return sharedEditor;
    }

    /**
     * Initializes a View for an Application.
     *
     * @param a The Application which holds the View.
     * @param p The View that needs to be initalized.
     */
    @Override
    public void initView(Application a, View p) {
        if (a.isSharingToolsAmongViews()) {
            ((AMView) p).setDrawingEditor(getSharedEditor());
        }
    }

    /**
     * Adds creation buttons on the specified JToolbar for the specified
     * DrawingEditor.
     * When a creation button is pressed, it sets a CreationTool on the
     * DrawingEditor. The user can then draw a Figure on a DrawingView using
     * the CreationTool.
     *
     * @param tb The toolbar.
     * @param editor The DrawingEditor.
     */
    private void addCreationButtonsTo(JToolBar tb, final DrawingEditor editor) {
        /*
         * The following hash map contains a default set of attributes.
         * All Figures that we create will use these attributes.
         */
        HashMap<AttributeKey, Object> attributes;
        attributes = new HashMap<AttributeKey, Object>();
        attributes.put(AttributeKeys.FILL_COLOR, ActivityDiagram.FILL_COLOR);
        attributes.put(AttributeKeys.STROKE_COLOR, ActivityDiagram.STROKE_COLOR);
        attributes.put(AttributeKeys.TEXT_COLOR, ActivityDiagram.TEXT_COLOR);

        // Different fill color definition for the InitialNodeFigure
        HashMap<AttributeKey, Object> initialNodeFigureAttributes;
        initialNodeFigureAttributes = new HashMap<AttributeKey, Object>();
        initialNodeFigureAttributes.put(AttributeKeys.FILL_COLOR,
                ActivityDiagram.INITIAL_NODE_FILL_COLOR);

        // Different fill color definition for the ActivityFinalNodeFigure.
        HashMap<AttributeKey, Object> activityFinalNodeFigureAttributes;
        activityFinalNodeFigureAttributes = new HashMap<AttributeKey, Object>();
        activityFinalNodeFigureAttributes.put(AttributeKeys.FILL_COLOR,
                ActivityDiagram.ACTIVITY_FINAL_NODE_FILL_COLOR);

        ResourceBundleUtil labels = ResourceBundleUtil.getBundle("ch.hslu.cm.am.Labels");
        ResourceBundleUtil drawLabels = ResourceBundleUtil.getBundle("org.jhotdraw.draw.Labels");

        ButtonFactory.addSelectionToolTo(tb, editor);
        tb.addSeparator();

        ButtonFactory.addToolTo(tb, editor, new CreationTool(
                new AMActionFigure(), attributes), "edit.createAction", labels);
        ButtonFactory.addToolTo(tb, editor, new CreationTool(
                new AMDecisionNodeFigure(), attributes), "edit.createDecisionNode", labels);
        ButtonFactory.addToolTo(tb, editor, new CreationTool(
                new AMInitialNodeFigure(), initialNodeFigureAttributes),
                "edit.createInitialNode", labels);
        ButtonFactory.addToolTo(tb, editor, new CreationTool(
                new AMActivityFinalNodeFigure(), activityFinalNodeFigureAttributes),
                "edit.createActivityFinalNode", labels);
        ButtonFactory.addToolTo(tb, editor, new ConnectionTool(
                new AMControlFlowFigure(), attributes), "edit.createControlFlow", labels);
        tb.addSeparator();
        ButtonFactory.addToolTo(tb, editor, new TextAreaCreationTool(
                new TextAreaFigure()), "edit.createTextArea", drawLabels);

    }

    /**
     * Creates toolbars for a View in the Application or for all Projects
     * in the Application.
     *
     * @param a The Application which holds the View.
     * @param pr The View that needs toolbars. This can be null, if the
     * application wishes to share the toolbars among all Projects!
     */
    @Override
    public List<JToolBar> createToolBars(Application a, View pr) {
        ResourceBundleUtil drawLabels = ResourceBundleUtil.getBundle("org.jhotdraw.draw.Labels");
        ResourceBundleUtil amLabels = ResourceBundleUtil.getBundle("ch.hslu.cm.am.Labels");
        ResourceBundleUtil cmLabels = ResourceBundleUtil.getBundle("ch.hslu.cm.Labels");
        AMView p = (AMView) pr;

        DrawingEditor editor;
        if (p == null) {
            editor = getSharedEditor();
        } else {
            editor = p.getDrawingEditor();
        }

        LinkedList<JToolBar> list = new LinkedList<JToolBar>();
        JToolBar tb;
        tb = new JToolBar();
        addCreationButtonsTo(tb, editor);
        tb.setName(drawLabels.getString("window.drawToolBar.title"));
        list.add(tb);
        tb = new JToolBar();
        ButtonFactory.addAttributesButtonsTo(tb, editor);
        tb.setName(drawLabels.getString("window.attributesToolBar.title"));
        list.add(tb);
        tb = new JToolBar();
        ButtonFactory.addAlignmentButtonsTo(tb, editor);
        tb.setName(drawLabels.getString("window.alignmentToolBar.title"));
        list.add(tb);
        tb = new JToolBar(cmLabels.getString("toolBarHTML"));
        HTMLToolBarFactory.addHTMLButtonsTo(tb);
        list.add(tb);
        return list;
    }

     @Override
    protected MenuBuilder createMenuBuilder() {
        AMMenuBuilder mb = new AMMenuBuilder();
        mb.setSuppressIcons(true);
        return mb;
    }
    @Override
    public URIChooser createOpenChooser(Application a, View v) {
        JFileURIChooser c = new JFileURIChooser();
        c.addChoosableFileFilter(new ExtensionFileFilter("Drawing .xml", "xml"));
        return c;
    }

    @Override
    public URIChooser createSaveChooser(Application a, View v) {
        JFileURIChooser c = new JFileURIChooser();
        c.addChoosableFileFilter(new ExtensionFileFilter("Drawing .xml", "xml"));
        return c;
    }

    @Override
    public URIChooser createExportChooser(Application a, View v) {
        JFileURIChooser c = new JFileURIChooser();
        c.addChoosableFileFilter(new ExtensionFileFilter("HTML .html", "html"));
        return c;
    }
}
